<?php

namespace App\Http\Controllers;

use App\Models\Client;
use App\Models\ExportLog;
use Illuminate\Http\Request;
use App\Services\PdfService;
use Illuminate\Support\Str;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Carbon;
use ZipArchive;

class PdfExportController extends Controller
{
    public function tracking(Client $client, PdfService $pdf)
    {
        $data = [
            'client' => $client,
            'entries' => $client->trackingEntries()->orderBy('date_signed_out')->get(),
        ];
        ExportLog::create(['user_id'=>Auth::id(),'client_id'=>$client->id,'mode'=>'tracking','created_at'=>Carbon::now()]);
        return $pdf->render('pdf.tracking-sheet', $data)
            ->download(sprintf('tracking-%s.pdf', Str::slug($client->client_name)));
    }

    public function followup(Client $client, PdfService $pdf)
    {
        $data = [
            'client' => $client,
            'comments' => $client->comments()->with('author')->orderByDesc('comment_date')->get(),
        ];
        ExportLog::create(['user_id'=>Auth::id(),'client_id'=>$client->id,'mode'=>'followup','created_at'=>Carbon::now()]);
        return $pdf->render('pdf.case-follow-up', $data)
            ->download(sprintf('followup-%s.pdf', Str::slug($client->client_name)));
    }

    public function full(Client $client, PdfService $pdf)
    {
        $data = [
            'client' => $client,
            'entries' => $client->trackingEntries()->orderBy('date_signed_out')->get(),
            'comments' => $client->comments()->with('author')->orderByDesc('comment_date')->get(),
        ];
        ExportLog::create(['user_id'=>Auth::id(),'client_id'=>$client->id,'mode'=>'full','created_at'=>Carbon::now()]);
        return $pdf->render('pdf.client-full', $data)
            ->download(sprintf('client-%s.pdf', Str::slug($client->client_name)));
    }

    public function batch(Request $request)
    {
        $request->validate([
            'client_ids'=>'required|array',
            'mode'=>'required|in:tracking,followup,full'
        ]);
        $ids = $request->input('client_ids');
        $mode = $request->input('mode');
        $folder = storage_path('app/exports/'.time());
        @mkdir($folder, 0777, true);
        $zipPath = storage_path('app/exports/batch-'+time()+'.zip');

        $zip = new ZipArchive();
        $zip->open($zipPath, ZipArchive::CREATE|ZipArchive::OVERWRITE);

        foreach ($ids as $id) {
            $client = Client::find($id);
            if(!$client) continue;
            $view = $mode === 'tracking' ? 'pdf.tracking-sheet' : ($mode==='followup' ? 'pdf.case-follow-up' : 'pdf.client-full');
            $pdf = app(\App\Services\PdfService::class)->render($view, [
                'client'=>$client,
                'entries'=>$client->trackingEntries,
                'comments'=>$client->comments()->with('author')->get(),
            ]);
            $filename = sprintf('%s-%s.pdf', $mode, \Illuminate\Support\Str::slug($client->client_name));
            $path = $folder.'/'.$filename;
            $pdf->save($path);
            $zip->addFile($path, $filename);
        }
        $zip->close();

        return response()->download($zipPath)->deleteFileAfterSend(true);
    }
}
